<?php
require_once __DIR__ . '../../vendor/autoload.php';
require_once ('../libs/rb.php');
require_once ('../libs/db.php');
require_once ('../libs/common.php');

$expectedToken = 'Bearer BcQyLywZRa86j5CvbtcjO';

 function updateSubscriptionExpiry($userId, $productId, $expiryDate) {
    $user = R::findOne('user', 'api_token = ?', [$userId]);

    R::begin();

    try {
        if (!$user) {
            error_log("User not found with apiToken: " . $userId);
            R::rollback();
            return;
        }

        $user->expirydate = $expiryDate;
        $userSubscription = R::findOne('usersubscriptions', 'user_id = ? AND product_id = ?', [$user->id, $productId]);
        
        if ($userSubscription) {
            $userSubscription->expirydate = $expiryDate;
            $userSubscription->expiry_date = $expiryDate;
            $userSubscription->subscription_status = 'expired';
            R::store($userSubscription);
        }

        R::store($user);
        R::commit();
    } catch (Exception $e) {
        // Handle any exceptions
        error_log("Error updating subscription expiry: " . $e->getMessage());
        R::rollback(); // Roll back the transaction
    }
}

function handleUserSubscription($userId, $expiryDate, $productId, $eventType) {
    $user = R::findOne('user', ' api_token = ? ', [$userId]);
    //error_log("Unhandled eventType: $eventType for userId: $userId with product: $productId and user->id: $user->id");
    if (!$user) {
        error_log("User not found with apitoken: " . $userId);
        return; // Early return if the user is not found
    }
                  
                  
    $userSubscription = R::findOne('usersubscriptions', ' user_id = ? AND product_id = ?', [$user->id, $productId]);
    $isNewSubscription = false;

    // Handle both new subscriptions and existing subscriptions
    if (!$userSubscription) {
        $userSubscription = R::dispense('usersubscriptions');
        $isNewSubscription = true;
    }
    
    $subscriptionPlan = R::findOne('subscriptionplans', ' product_id = ? ', [$productId]);
    if($subscriptionPlan){
        // Set or update user subscription details
        $userSubscription->user_id = $user->id;
        $userSubscription->product_id = $productId;
        $userSubscription->expirydate = $expiryDate;
        $userSubscription->expiry_date = $expiryDate;
        $user->expirydate = $expiryDate;
        $userSubscription->credits_per_period = $subscriptionPlan->credits;
    
        // Initialize or update subscription start and last credit update dates for certain event types
        if ($isNewSubscription || $eventType === 'INITIAL_PURCHASE' || $eventType === 'RENEWAL') {
            $userSubscription->subscription_start = date('Y-m-d H:i:s');
            $userSubscription->last_credit_update = date('Y-m-d H:i:s');
            $userSubscription->total_credits = $userSubscription->total_credits + $subscriptionPlan->credits;
            $userSubscription->subscription_status = 'active';
        }
        else if($eventType === 'CANCELLATION') {
             $userSubscription->subscription_status = 'cancelled';
        }
    
        // Save the subscription details
        R::store($user);
        R::store($userSubscription);
    }

    // Optionally log the action for auditing
    error_log("Handled subscription for user (API Token: $userId) with product ID: $productId, Event: $eventType");

    // Update credits only for INITIAL_PURCHASE and RENEWAL events
    if ($eventType === 'INITIAL_PURCHASE' || $eventType === 'RENEWAL') {
        updateUserSubscriptionCredits($user->id, $productId, $eventType);
    }
}


function updateUserSubscriptionCredits($userId, $productId, $eventType) {
    $subscriptionPlan = R::findOne('subscriptionplans', ' product_id = ? ', [$productId]);
    $user = R::findOne('user', ' id = ? ', [$userId]);

    if (!$user) {
        // Improved error logging
        error_log("User not found with ID: $userId", 3, 'weberror.log');
        return false; // Indicate failure
    }

    if (!$subscriptionPlan) {
        // Improved error logging
        error_log("Subscription plan not found for product ID: $productId", 3, 'weberror.log');
        return false; // Indicate failure
    }

    try {
        R::begin();
        $user->reveal_count += $subscriptionPlan->credits;
        $creditLog = R::dispense('creditlog');
        $creditLog->user_id = $userId;
        $creditLog->event_type = $eventType;
        $creditLog->credits_added = $subscriptionPlan->credits;
        $creditLog->product_id = $productId;
        R::store($creditLog);
        R::store($user);
        R::commit();
        return true; // Indicate success
    } catch (Exception $ex) {
        R::rollback();
        // Log exception details
        error_log("Failed to update user subscription credits: " . $ex->getMessage(), 3, 'weberror.log');
        return false; // Indicate failure
    }
}

/*
function updateUserCredits($userId, $productId) {
    $user = R::findOne('user', ' api_token = ? ', [$userId]);
    if (!$user) {
        error_log("User not found with apitoken: " . $userId);
        return; // Exit the function if no user is found
    }
    
    $creditPack = R::findOne('creditpacks', ' product_id = ? ', [$productId]);
    if (!$creditPack) {
        error_log("Credit pack not found for product ID: " . $productId);
        return; // Exit the function if no credit pack is found
    }

    // Update the user's credits
    $user->credits += $creditPack->credits;
    R::store($user);
    
    // Log the credit update in the database
    $creditLog = R::dispense('creditlog');
    $creditLog->user_id = $user->id; // Assuming the user ID is needed, not the api_token
    $creditLog->credits_added = $creditPack->credits;
    $creditLog->product_id = $productId;
    $creditLog->created_at = date('Y-m-d H:i:s'); // Optional: Track when the update was made
    R::store($creditLog);

    // Optional: Log successful update for auditing
    error_log("Updated user (API Token: $userId) credits by " . $creditPack->credits . "; Logged in creditlog.");
}*/


$headers = apache_request_headers();
$receivedToken = $headers['Authorization'] ?? '';

if ($receivedToken !== $expectedToken) {
    http_response_code(401); // Unauthorized
    exit('Invalid authorization header: received: '.$receivedToken);
}

$payload = file_get_contents('php://input');
$data = json_decode($payload);

if ($data === null) {
    http_response_code(400);
    echo 'Invalid JSON';
    exit;
}

$eventType = $data->event->type ?? '';
$userId = $data->event->app_user_id ?? '';
$productId = $data->event->product_id ?? '';
$expirationAtMs = $data->event->expiration_at_ms ?? '';
$expiryDate = $expirationAtMs ? date('Y-m-d H:i:s', $expirationAtMs / 1000) : null;

// Process the event based on its type
switch ($eventType) {
    case 'NON_RENEWING_PURCHASE':
        // Handle non-renewing purchase
        //updateUserCredits($userId, $productId);
        break;
    case 'INITIAL_PURCHASE':
    case 'RENEWAL':
    case 'UNCANCELLATION':
    case 'TRANSFER':
    case 'CANCELLATION':
        // Handle subscription-related events
        if ($expiryDate) {
            handleUserSubscription($userId, $expiryDate, $productId, $eventType);
        } else {
            error_log("Expiry Date not found for userId: $userId and eventType: $eventType");
        }
        break;
    case 'EXPIRATION':
            $expiryDate = date('Y-m-d H:i:s'); // Consider if this fallback is appropriate
            updateSubscriptionExpiry($userId, $productId, $expiryDate);
        break;
    default:
        // Log unhandled event types
        error_log("Unhandled eventType: $eventType for userId: $userId");
        break;
}



http_response_code(200); // OK
echo 'Webhook processed';

?>
