<?php
require_once 'constants.php';
require_once 'common.php';

class Database {
    private static $instance = null;
    private static $redis = null;
    private static $databaseInitialized = false;

    // Private constructor to prevent direct instantiation
    private function __construct() {}

    // Get or create the singleton instance
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    // Initialize MySQL Database using RedBeanPHP (only when needed)
    public function getDatabase() {
        if (!self::$databaseInitialized) {
            try {
                if (IN_DEV_ENVIRONMENT) {
                    R::setup('mysql:host=' . MYSQL_HOST_DEV . ';port=' . MYSQL_PORT_DEV . ';dbname=' . MYSQL_DBNAME_DEV . ';charset=utf8', MYSQL_USERNAME_DEV, MYSQL_PASSWORD_DEV);
                } else {
                    R::setup('mysql:host=' . MYSQL_HOST_PROD . ';port=' . MYSQL_PORT_PROD . ';dbname=' . MYSQL_DBNAME_PROD . ';charset=utf8', MYSQL_USERNAME_PROD, MYSQL_PASSWORD_PROD);
                }
                R::exec("SET time_zone='+00:00'");
                R::freeze(true);

                if (!R::testConnection()) {
                    throw new Exception("Failed to connect to MySQL database.");
                }
                self::$databaseInitialized = true;
            } catch (Exception $e) {
                error_log("Database connection error: " . $e->getMessage());
                throw $e;
            }
        }
        return R::getDatabaseAdapter();
    }

    // Initialize Redis connection (only when needed)
    public function getRedis() {
        if (self::$redis === null) {
            try {
                self::$redis = new Redis();
                self::$redis->pconnect(REDIS_PREFIX . REDIS_HOST, REDIS_PORT, REDIS_TIMEOUT);
                self::$redis->setOption(Redis::OPT_READ_TIMEOUT, -1);
                self::$redis->auth(['default', REDIS_PASSWORD]);
                if (!self::$redis->ping()) {
                    throw new Exception("Failed to ping Redis server.");
                }
            } catch (RedisException $e) {
                error_log("Redis connection error: " . $e->getMessage());
                throw $e;
            }
        }
        return self::$redis;
    }

    // Disallow cloning and unserialization which can create multiple instances
    private function __clone() {}
    public function __wakeup() {}
}

// Usage
try {
    $dbInstance = Database::getInstance();
    $db = $dbInstance->getDatabase();  // Initializes MySQL connection when needed
    $redis = $dbInstance->getRedis();  // Initializes Redis connection when needed

    // Example Redis operation
    //$redis->set('test', 'Hello World!');
    //echo $redis->get('test');
} catch (Exception $e) {
    die("Initialization failed: " . $e->getMessage());
}
?>
