<?php
require_once __DIR__ . '/../vendor/autoload.php';
require_once 'rb.php';
require_once 'constants.php';

use Aws\S3\S3Client;
use Aws\Credentials\CredentialProvider;

use Monolog\Logger;
use Monolog\Handler\StreamHandler;
use Monolog\Processor\IntrospectionProcessor;

class AppLogger {
    private static $logger = null;

    private static function getLogger() {
        if (self::$logger === null) {
            self::$logger = new Logger('app');

            // Log to a file, for example
            self::$logger->pushHandler(new StreamHandler(__DIR__ . '/../error_logs/app.log', Logger::DEBUG));

            // Automatically add file name, line number, etc., to each log entry
            //self::$logger->pushProcessor(new IntrospectionProcessor());

            // You can push more handlers or processors here as needed
        }
        return self::$logger;
    }

    public static function logError($message, $context = []) {
        self::getLogger()->error($message, $context);
    }

    public static function logInfo($message, $context = []) {
        self::getLogger()->info($message, $context);
    }

    public static function logWarning($message, $context = []) {
        self::getLogger()->warning($message, $context);
    }

    // You can add more logging level methods here...
}

function subscriptionIsActive($userId) {

    $now = new DateTime("now", new DateTimeZone("UTC"));
    try {
        $user = R::load('user', $userId);

        if (!$user->id) {
            return false;
        }

        $expiryDate = new DateTime($user->expirydate, new DateTimeZone("UTC"));
        return $expiryDate > $now;

    } catch (Exception $e) {
        error_log($e->getMessage());
        return false;
    }
}



function deleteObjectFromR2($bucket, $keyName, $provider=null) {
    if ($provider == null) {
        $provider = CredentialProvider::ini(null, CLOUDFLARE_R2_CREDENTIALS_PATH);
        $provider = CredentialProvider::memoize($provider);
    }

    $s3 = new S3Client([
        'endpoint' => CLOUDFLARE_R2_ENDPOINT,
        'region' => 'auto',
        'version' => 'latest',
        'credentials' => $provider,
        'use_path_style_endpoint' => true,
    ]);

    try {
        $s3->deleteObject([
            'Bucket' => $bucket,
            'Key'    => $keyName,
        ]);
        return "Object deleted successfully.";
    } catch (AwsException $e) {
        // Handle the error
        return "Error deleting object: " . $e->getMessage();
    }
}

function generateUploadPresignedURL($bucket, $keyName, $expires, $metadata, $provider=null) {
    
    if($provider == null){
        $provider = CredentialProvider::ini(null, CLOUDFLARE_R2_CREDENTIALS_PATH);
        $provider = CredentialProvider::memoize($provider);
    }
    
        $s3 = new S3Client([
            'endpoint' => CLOUDFLARE_R2_ENDPOINT,
            'region' => 'auto',
            'version' => 'latest',
			'credentials' => $provider,
            'use_path_style_endpoint' => true
        ]);

        $cmd = $s3->getCommand('PutObject', [
            'Bucket' => $bucket,
            'Key' => $keyName,
            'Metadata' => $metadata,
            'ContentType' => 'image/jpeg'
        ]);

        $request = $s3->createPresignedRequest($cmd, $expires);
        return (string) $request->getUri();
    }
    
    function getAccessToken($serviceAccountJsonFile) {
        $serviceAccount = json_decode(file_get_contents($serviceAccountJsonFile), true);
        $client = new Google_Client();
        $client->setAuthConfig($serviceAccount);
        $client->addScope('https://www.googleapis.com/auth/firebase.messaging');
        $token = $client->fetchAccessTokenWithAssertion();
        return $token["access_token"];
    }

function sendPushNotification($fcm_token, $title, $message, $serviceAccountJsonFile) {
    $accessToken = getAccessToken($serviceAccountJsonFile);
    $url = 'https://fcm.googleapis.com/v1/projects/yaaas-2fe8a/messages:send';
    $fields = [
        'message' => [
            'token' => $fcm_token,
            'notification' => [
                'title' => $title,
                'body' => $message
            ]
        ]
    ];
    $headers = [
        'Authorization: Bearer ' . $accessToken,
        'Content-Type: application/json'
    ];

    $retryCount = 0;
    $maxRetries = 5;
    $retryDelay = 1;  // Start with 1 second

    while ($retryCount <= $maxRetries) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($fields));
        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode == 200) {
            return true; // Successfully sent
        } else if ($httpCode != 429 && $httpCode >= 400) {
            break;  // Break on client and server errors except for rate limits
        }

        if ($retryCount < $maxRetries) {
            sleep($retryDelay);
            $retryDelay *= 2;  // Double the delay for next retry
        }
        $retryCount++;
    }

    return false;  // Return false if failed to send after retries
}


    function enqueueNotification($userId, $friendId, $notificationType) {
        $redis = Database::getRedis();
        if ($redis === null) {
            error_log("Failed to retrieve Redis instance.");
            return false;  // Early return if Redis is not available
        }
    
        $payload = json_encode([
            'user_id' => $userId,
            'friend_id' => $friendId,
            'notification_type' => $notificationType
        ]);
    
        try {
            $redis->lPush("notification_queue", $payload);
            return true;
        } catch (RedisException $e) {
            error_log("Failed to enqueue notification: " . $e->getMessage());
            return false;
        }
    }
    
    
    function getIP() { 
        $ip; 
        if (getenv("HTTP_CLIENT_IP")) 
        $ip = getenv("HTTP_CLIENT_IP"); 
        else if(getenv("HTTP_X_FORWARDED_FOR")) 
        $ip = getenv("HTTP_X_FORWARDED_FOR"); 
        else if(getenv("REMOTE_ADDR")) 
        $ip = getenv("REMOTE_ADDR"); 
        else 
        $ip = "UNKNOWN";
        return $ip; 
    }
    
    function generateUUIDv4() {
        // Generate 16 bytes (128 bits) of random data or use the openssl_random_pseudo_bytes() function.
        $data = random_bytes(16);
    
        // Set the version to 0100 (4 in binary).
        $data[6] = chr(ord($data[6]) & 0x0f | 0x40); // Set version to 0100 (UUID version 4)
    
        // Set the bits 6-7 to 10 (2 in binary).
        $data[8] = chr(ord($data[8]) & 0x3f | 0x80); // Set bits 6-7 to 10
    
        // Output the 36 character UUID.
        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }
    
    function randString($length,$type=false,$exclude=false,$repeat=true)
    {
      $string['lower'] = 'abcdefghijklmnopqrstuvwxyz';
      $string['upper'] = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
      $string['digits'] = '0123456789';
      $type=strtolower($type);
    
      if($exclude)
      {
        $exclude = str_split($exclude);
        foreach($string as $key=>$value)
        {
          $string[$key] = str_ireplace($exclude,'',$value);
        }
      }
    
      switch ($type)
      {
        case 'numeric':
          $chars = $string['digits'];
          break;
        case 'alpha':
          $chars = $string['lower'].$string['upper'];
          break;
        case 'alpha-uc':
          $chars = $string['upper'];
          break;
        case 'alpha-lc':
          $chars = $string['lower'];
          break;
        case 'mixed-uc':
          $chars = $string['upper'].$string['digits'];
          break;
        case 'mixed-lc':
          $chars = $string['lower'].$string['digits'];
          break;
        case 'mixed':
        default:
          $chars=$string['lower'].$string['upper'].$string['digits'];
          break;
      }
    	$char_length = strlen($chars);
    	if((!$repeat) AND ($length > $char_length ))
      {
        $err_msg = "ALLOWING REPEATS - only $char_length chars available ";
        $repeat=true;
        trigger_error($err_msg,E_USER_NOTICE);
      }
    
      $rand_string = "";
      for ($i = 0; ($i < $length); $i++)
      {
    		$char = substr($chars,mt_rand(0, strlen($chars) - 1), 1);
        $rand_string .= $char;
        if(!$repeat)$chars=str_ireplace($char,'',$chars);
      }
      return $rand_string;
    }
    
    function randomStr($min_chars = 4, $max_chars = 15, $use_chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789') 
    { 
        $num_chars  = rand($min_chars, $max_chars); 
        $num_usable = strlen($use_chars) - 1; 
        $string     = ''; 
    
        for($i = 0; $i < $num_chars; $i++) 
        { 
            $rand_char = rand(0, $num_usable); 
    
            $string .= $use_chars[$rand_char]; 
        } 
    
        return $string; 
    }  
    
     function generate9DigitNumber($inputInteger) {
        // Convert the integer to a string and hash it
        $hash = md5(strval($inputInteger));
    
        // Convert a portion of the hash to a 9-digit number
        $number = hexdec(substr($hash, 0, 8)) % 1000000000;
    
        return $number;
    }
    

//we should check expiryDate against mysql
//server time so we know timezones are same
//but problem now is that have already set in our db.php
//timezone in mysql to be UTC before we write expirydate
//so we should just remove that before enabling this
/*function hasActiveSubscription($userId) {
   
    $user = R::load('user', $userId);

    if ($user->id) {
        $expiryDate = $user->expirydate;
        $mysqlTime = R::getCell('SELECT NOW()');

        if (!empty($expiryDate)) {
            if (strtotime($expiryDate) > strtotime($mysqlTime)) {
                return true; // Subscription is active
            }
        } else if ($user->lifetime == 1) {
            return true; // Lifetime subscription is active
        }
    }

    return false; // No active subscription
}*/

 function resizeImage($file, $w, $h) {
        // Get the current dimensions and file type of the image
        list($width, $height, $type) = getimagesize($file);
        
        // Determine the image type (JPEG, PNG, etc.) and create the source image
        switch ($type) {
            case IMAGETYPE_JPEG:
                $src = imagecreatefromjpeg($file);
                break;
            case IMAGETYPE_PNG:
                $src = imagecreatefrompng($file);
                break;
            default:
                return; // Unsupported file type
        }
        
        // Check if the image is already smaller than the desired dimensions
        if ($width <= $w && $height <= $h) {
            return; // No need to resize
        }
        
        // Calculate new dimensions to maintain aspect ratio
        $ratio = $width / $height;
        if ($w / $h > $ratio) {
            $newWidth = $h * $ratio;
            $newHeight = $h;
        } else {
            $newHeight = $w / $ratio;
            $newWidth = $w;
        }
        
        // Create a new image at the new dimensions
        $dst = imagecreatetruecolor($newWidth, $newHeight);
        imagecopyresampled($dst, $src, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
        
        // Save the resized image as JPEG
        // Ensure the filename extension is .jpg
        $file = preg_replace('/\.[^.]+$/', '.jpg', $file);
        imagejpeg($dst, $file, 90); // 90 is the quality setting
        
        // Free resources
        imagedestroy($src);
        imagedestroy($dst);
    }

function hasActiveSubscription($userId) {
    // Fetch the user from the database
    $user = R::load('user', $userId); // Assuming 'user' is your table name and $userId is the user's ID

    if ($user->id) { // Check if the user exists
        $expiryDate = $user->expirydate; // Assuming 'expirydate' is the field name in your table

        // Check if the expiry date is set and in the future
        if (!empty($expiryDate) && strtotime($expiryDate) > time()) {
            return true; // Subscription is active
        }
        else if($user->lifetime == 1){
            return true;
        }
    }

    return false; // No active subscription
}

?>