<?php
require_once __DIR__ . '/../../vendor/autoload.php';
require_once ('../libs/rb.php');
require_once ('../libs/db.php');
require_once ('../libs/common.php');

class poll {
    //public $db;
    function __construct() {

    }
    
    function isFirstQuestion($poll_id, $question_id) {
        $firstQuestion = R::getRow('SELECT * FROM pollquestion WHERE poll_id = ? ORDER BY sortorder ASC, id ASC LIMIT 1', [$poll_id]);
    
        return $firstQuestion && $firstQuestion['question_id'] == $question_id;
    }
    
    function isLastQuestion($poll_id, $question_id) {
        $lastQuestion = R::getRow('SELECT * FROM pollquestion WHERE poll_id = ? ORDER BY sortorder DESC, id DESC LIMIT 1', [$poll_id]);
    
        return $lastQuestion && $lastQuestion['question_id'] == $question_id;
    }
    
    /**
     * Validates the existence of user and throws exception if not found.
     */
    private function validateUser($user) {
        if (!$user) {
            throw new RestException(400, 'Error: User account not found.');
        }
    }

    /**
     * Validates required inputs and throws an exception if any are missing.
     */
    private function validateInputs($inputs) {
        foreach ($inputs as $key => $value) {
            if (!$value) {
                throw new RestException(400, "Error: $key must be supplied.");
            }
        }
    }
        
    protected function delete($id, $name, $data, $tag) {
        throw new RestException(501, 'DELETE not implemented');
    }
    
    private function validateFriendExists($friend_id, $poll_id) {
        $friendExists = R::findOne('user', 'id = ?', [$friend_id]);
        if (!$friendExists) {
            AppLogger::logError("[poll]: friend was not found vote skipped: ", [
                'firebaseuserid' => $firebaseuserid,
                'pollid' => $poll_id,
                'friendid' => $friend_id
                  ]);
                  
            throw new Exception('Friend does not exist, vote skipped.');
        }
    }
    
    private function recordResponse($userid, $user, $poll_id, $question_id, $poll_question_id, $friend_id, $option1, $option2, $option3, $option4) {
    
        $userpoll = R::findOne('userpoll', ' user_id = ? AND poll_id = ? ', [$userid, $poll_id]);
        if($userpoll){
            $userpoll->coins_earned = $userpoll->coins_earned + 1;
            R::store($userpoll);
        }
    
        $response = R::dispense('response');
        $response->user_id = $userid;
        $response->gender_id = $user->gender_id;
        $response->grade = $user->grade;
        $response->question_id = $question_id;
        $response->poll_question_id = $poll_question_id;
        $response->selected_friend_id = $friend_id;
        $response->option1 = $option1;
        $response->option2 = $option2;
        $response->option3 = $option3;
        $response->option4 = $option4;
        R::store($response);
    }
    
    private function fetchOrCreateUserPoll($userid, $poll_id, $question_id) {
        $userpoll = R::findOne('userpoll', ' user_id = ? AND poll_id = ? ', array($userid, $poll_id));
        
        if(!$userpoll){
            $userpoll = R::dispense('userpoll');
            $userpoll->user_id = $userid;
            $userpoll->poll_id = $poll_id;
            
             AppLogger::logError("[poll]: userpoll was not found, this should not happen: ", [
            'firebaseuserid' => $firebaseuserid,
            'questionid' => $question_id,
            'pollid' => $poll_id
              ]);
        }
        
        if($this->isLastQuestion($poll_id, $question_id)){
            $userpoll->completed = true; 
            $userpoll->end_time = gmdate("Y-m-d H:i:s");
        }
        else {
            $userpoll->current_question_id = (string) ($question_id + 1);
        }
        

        R::store($userpoll);
        return $userpoll;
    }
    
    private function fetchNextQuestionAnswers($nextQuestion, $userid) {
      if ($nextQuestion) {
          $visibilityBoost = R::getRow("SELECT id, user_id FROM visibilityboost WHERE target_user_id = ? AND applied = 0 LIMIT 1", [$userid]);

            $totalFriendsCount = R::getCell("SELECT COUNT(*) FROM friendship WHERE user_id = ?", [$userid]);
            $offset = $totalFriendsCount > 200 ? rand(0, $totalFriendsCount - 200) : 0;
            $friendIDs = R::getCol("
                SELECT 
                    u.id
                FROM 
                    friendship AS f
                JOIN 
                    user AS u ON f.friend_id = u.id
                WHERE 
                    f.user_id = ?
                LIMIT 200 OFFSET ?", 
                [$userid, $offset]);
          

           $selectedFriends = [];
            if (!empty($friendIDs)) {
                $selectedFriendIDs = (count($friendIDs) > 4) ? array_rand(array_flip($friendIDs), 4) : $friendIDs;
                $selectedFriends = R::getAll("SELECT u.id as friend_id, u.first_name, u.last_name, COALESCE(rq.response_count, 0) as response_count
                                                FROM user AS u
                                                LEFT JOIN responsecount rq ON rq.friend_id = u.id AND rq.question_id = ?
                                                WHERE u.id IN (" . implode(',', $selectedFriendIDs) . ")",
                                                [$nextQuestion['id']]);
            }

            // Include boost friend if applicable
            if (!empty($visibilityBoost)) {
                
                $boostFriendDetails = R::getRow("SELECT u.id as friend_id, u.first_name, u.last_name 
                                                FROM user AS u
                                                WHERE u.id = ?", [$visibilityBoost['user_id']]);
                array_unshift($selectedFriends, $boostFriendDetails);
                // Mark visibility boost as applied
                R::exec("UPDATE visibilityboost SET applied = 1 WHERE id = ?", [$visibilityBoost['id']]);
            }

          
          
            $nextQuestion['friends'] = $selectedFriends;
    
        } else {
            //there is no next question so...
        }
            
        return $nextQuestion;
    }
    
    private function fetchNextQuestion($userid, $pollid, $currentQuestionID) {
        $nextQuestion = null;
    
        $userpoll = R::findOne('userpoll', ' userid = ? AND pollid = ? ', [$userid, $pollid]);
        
        
    
        if ($pollid && $userpoll->completed == 0) {
            
            // Get the sort order of the current question
            $currentSortOrder = R::getCell(
                "SELECT sortorder FROM pollquestion WHERE question_id = ? AND poll_id = ?",
                [$currentQuestionID, $pollid]
            );
    
            // Check if currentSortOrder is not null, then proceed to get the next question
            if ($currentSortOrder !== null) {
                // Fetch the next question based on the sort order
                $nextQuestionSQL = "
                    SELECT pq.question_id, pq.id AS poll_question_id, q.id, q.text, q.emoji, pq.sortorder
                    FROM pollquestion AS pq
                    JOIN question AS q ON pq.question_id = q.id
                    WHERE pq.poll_id = ? AND pq.sortorder > ?
                    ORDER BY pq.sortorder ASC
                    LIMIT 1";
    
                $nextQuestion = R::getRow($nextQuestionSQL, [$pollid, $currentSortOrder]);
            } else {
                // If we don't have a current sort order, then this user has not answered any question yet,
                // so we fetch the first question.
                $nextQuestionSQL = "
                    SELECT pq.question_id, pq.id AS poll_question_id, q.id, q.text, q.emoji, pq.sortorder
                    FROM pollquestion AS pq
                    JOIN question AS q ON pq.question_id = q.id
                    WHERE pq.poll_id = ?
                    ORDER BY pq.sortorder ASC
                    LIMIT 1";
    
                $nextQuestion = R::getRow($nextQuestionSQL, [$pollid]);
            }
        }
    
        return $nextQuestion;
    }

        
    protected function put($poll_id, $friend_id, $question_id, $poll_question_id,$option1, $option2, $option3, $option4) {
        
        try{
            
            R::begin();
            $firebaseuserid = BasicAuth::getfirebaseuserid();
            $userid = BasicAuth::getuserid();
            $user = BasicAuth::getuser();
            
            $responses = [];
            $this->validateUser($user);
            $this->validateInputs(compact('firebaseuserid', 'poll_id', 'question_id', 'friend_id', 'poll_question_id', 'option1', 'option2', 'option3', 'option4'));
            $this->validateFriendExists($friend_id, $poll_id);

            $this->recordResponse($userid, $user, $poll_id, $question_id, $poll_question_id, $friend_id, $option1, $option2, $option3, $option4);

            
            
            $userpoll = $this->fetchOrCreateUserPoll($userid, $poll_id, $question_id);
            
            $nextQuestion = null;
     
        
            if($userpoll->completed == false) {
                $nextQuestion = $this->fetchNextQuestion($userid, $poll_id, $question_id);
                
               if ($nextQuestion) {
                    $nextQuestion = $this->fetchNextQuestionAnswers($nextQuestion, $userid);
                }
            }
            
            $userpoll->nextQuestion = $nextQuestion;
            $responses = ["userpoll" => $userpoll];
            

            R::commit();
            return ['status' => 'success', 'responses' => $responses];
        } catch (Exception $e) {
            R::rollback();
             AppLogger::logError("[poll]: General error in post method: " . $e->getMessage(), [
                'firebaseuserid' => $firebaseuserid
                  ]);
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
                    
    }
    
    protected function post($poll_id, $current_question_id){
        
        try {
            $firebaseuserid = BasicAuth::getfirebaseuserid();
            $userid = BasicAuth::getuserid();
            $user = BasicAuth::getuser();
    
            // Attempt to find an existing poll
            try {
                $existingPoll = R::findOne('userpoll', 'user_id = ? AND completed = 0 ORDER BY id DESC LIMIT 1', [$userid]);
    
                if ($existingPoll) {
                    $existingPoll->current_question_id = $current_question_id;
                    if($this->isLastQuestion($poll_id, $current_question_id)){
                        $existingPoll->end_time = gmdate("Y-m-d H:i:s");
                        $existingPoll->completed = 1; 
                    }
                    R::store($existingPoll);
                    return $existingPoll;
                }
            } catch (Exception $e) {
                // Log error for existing poll processing
                AppLogger::logError("[poll]: Error updating existing poll: " . $e->getMessage(), [
                    'firebaseuserid' => $firebaseuserid,
                    'poll_id' => $poll_id,
                    'current_question_id' => $current_question_id
                ]);
            }
    
            // If no existing poll, create a new one
            try {
                $newPoll = R::dispense('userpoll');
                $newPoll->user_id = $userid;
                $newPoll->poll_id = $poll_id;
                $newPoll->start_time = gmdate("Y-m-d H:i:s");
                $newPoll->end_time = NULL;
                $newPoll->completed = 0;
                $newPoll->current_question_id = $current_question_id;
                $newPoll->active = 1;
                R::store($newPoll);
                return $newPoll;
            } catch (Exception $e) {
                // Log error for new poll creation
                AppLogger::logError("[poll]: Error creating new poll: " . $e->getMessage(), [
                    'firebaseuserid' => $firebaseuserid,
                    'poll_id' => $poll_id,
                    'current_question_id' => $current_question_id
                ]);
                throw new RestException(500, "An error occurred while processing the poll.");
            }
        } catch (Exception $e) {
            // General catch block for other unexpected errors
            AppLogger::logError("[poll]: General error in post method: " . $e->getMessage(), [
                'firebaseuserid' => $firebaseuserid
            ]);
            throw new RestException(500, "An error occurred while processing the request.");
        }
    }
    
    
    protected function findLatestUserPollByUserId($userId) {
        try {
            return R::findOne('userpoll', ' user_id = ? ORDER BY poll_id DESC LIMIT 1', [$userId]);
        } catch (Exception $e) {
             AppLogger::logError("[poll]: Error finding latest user poll: " . $e->getMessage(), [
                    'userid' => $userId
                ]);
            throw new RestException(500, "Error finding latest user poll.");
        }
    }
    
    protected function createNewUserPoll($userId, $pollId) {
        try {
            $pollquestion = R::findOne('pollquestion', ' poll_id = ? ORDER BY sortorder ASC LIMIT 1 ', [$pollId]);
            if($pollquestion){
                $userpoll = R::dispense('userpoll');
                $userpoll->user_id = $userId;
                $userpoll->poll_id = $pollId;
                // $userpoll->start_time = gmdate("Y-m-d H:i:s");
                $userpoll->current_question_id = $pollquestion->question_id;
                $id = R::store($userpoll);
        
                if (!$id) {
                    throw new RestException(500, "Failed to create a new user poll.");
                }
        
                return R::load('userpoll', $id);
            }
            else {
                 throw new RestException(500, "Failed to find pollquestions.");
            }
        } catch (Exception $e) {
             AppLogger::logError("[poll]: Error creating new user poll: " . $e->getMessage(), [
                    'userid' => $userId,
                    'pollid' => $pollId
                ]);
            throw new RestException(500, "Error creating new user poll.");
        }
    }
    
    protected function get() {
    	
    	try {
            $userId = BasicAuth::getUserId();
            $latestUserPoll = $this->findLatestUserPollByUserId($userId);
            $currentTime = new DateTime("now", new DateTimeZone('UTC'));
    
            if (!$latestUserPoll) {
                $latestUserPoll = $this->createNewUserPoll($userId, 1);
            } elseif ($latestUserPoll->completed) { 
                $endTime = new DateTime($latestUserPoll->end_time, new DateTimeZone('UTC'));
                $timeDiff = $currentTime->getTimestamp() - $endTime->getTimestamp();
    
                // Check if 60 minutes have passed
                if ($timeDiff >= 3600) { // 3600 seconds = 60 minutes
                    $latestUserPoll = $this->createNewUserPoll($userId, $latestUserPoll->poll_id + 1);
                } else {
                    // Not enough time has passed, consider how you want to handle this case.
                    // For example, you might want to return the current poll, a message indicating
                    // the time remaining until the next poll, or null.
                    // This example will return the latestUserPoll without creating a new one.
                    // Optionally, add logic here to calculate remaining time and attach it to the response.
                }
            }
    
            if ($latestUserPoll->completed == 0) {
                $current_question_id = $latestUserPoll->current_question_id;
                $nextQuestion = $this->fetchNextQuestion($userId, $latestUserPoll->poll_id, $current_question_id-1);
                    
               if ($nextQuestion) {
                    $nextQuestion = $this->fetchNextQuestionAnswers($nextQuestion, $userId);
                }
               
                $latestUserPoll->nextQuestion = $nextQuestion;
            }
            else{
                //could not find user poll
            }
    
            return $latestUserPoll;
        } catch (Exception $e) {
             AppLogger::logError("[poll]: Failed to get or create user poll: " . $e->getMessage(), [
                    'userid' => $userId
                ]);
                
            throw new RestException(500, "Failed to get or create user poll.");
        }
    }
}
?>